#include <WiFi.h>
#include <ESPAsyncWebServer.h>
#include <LittleFS.h>
#include <ArduinoJson.h>
#include <time.h>

// --- CONFIGURATION ---
const char *ssid = "......";
const char *password = "......";
const int PR1RELAY_PIN = 26;
const int PR2RELAY_PIN = 25;
const int PR3RELAY_PIN = 33;
const int PR4RELAY_PIN = 32;
const char *TZ_INFO = "CET-1CEST,M3.5.0,M10.5.0/3";

AsyncWebServer server(80);

// Variables de contrôle
String PR1heureDebut = "08:00";
String PR1heureFin = "18:00";
bool PR1modeAuto = true;
bool PR1relayState = false;

String PR2heureDebut = "09:00";
String PR2heureFin = "19:00";
bool PR2modeAuto = true;
bool PR2relayState = false;

String PR3heureDebut = "10:00";
String PR3heureFin = "20:00";
bool PR3modeAuto = true;
bool PR3relayState = false;

String PR4heureDebut = "11:00";
String PR4heureFin = "21:00";
bool PR4modeAuto = true;
bool PR4relayState = false;

String now;

// --- SYSTÈME DE SAUVEGARDE LITTLEFS ---

void saveSettings() {
  File file = LittleFS.open("/config.json", "w");
  if (!file) {
    Serial.println("Erreur lors de l'ouverture du fichier pour écriture");
    return;
  }
  StaticJsonDocument<256> doc;
  doc["PR1debut"] = PR1heureDebut;
  doc["PR1fin"] = PR1heureFin;
  doc["PR1auto"] = PR1modeAuto;
  doc["PR1etatMan"] = PR1relayState;

  doc["PR2debut"] = PR2heureDebut;
  doc["PR2fin"] = PR2heureFin;
  doc["PR2auto"] = PR2modeAuto;
  doc["PR2etatMan"] = PR2relayState;

  doc["PR3debut"] = PR3heureDebut;
  doc["PR3fin"] = PR3heureFin;
  doc["PR3auto"] = PR3modeAuto;
  doc["PR3etatMan"] = PR3relayState;

  doc["PR4debut"] = PR4heureDebut;
  doc["PR4fin"] = PR4heureFin;
  doc["PR4auto"] = PR4modeAuto;
  doc["PR4etatMan"] = PR4relayState;

  if (serializeJson(doc, file) == 0) {
    Serial.println("Erreur d'écriture JSON");
  }
  file.close();
  Serial.println("Paramètres sauvegardés dans LittleFS");
}

void loadSettings() {
  if (LittleFS.exists("/config.json")) {
    File file = LittleFS.open("/config.json", "r");
    StaticJsonDocument<256> doc;
    DeserializationError error = deserializeJson(doc, file);
    if (!error) {
      PR1heureDebut = doc["PR1debut"] | "08:00";
      PR1heureFin = doc["PR1fin"] | "18:00";
      PR1modeAuto = doc["PR1auto"] | true;
      PR1relayState = doc["PR1etatMan"] | false;

      PR2heureDebut = doc["PR2debut"] | "09:00";
      PR2heureFin = doc["PR2fin"] | "19:00";
      PR2modeAuto = doc["PR2auto"] | true;
      PR2relayState = doc["PR2etatMan"] | false;

      PR3heureDebut = doc["PR3debut"] | "10:00";
      PR3heureFin = doc["PR3fin"] | "20:00";
      PR3modeAuto = doc["PR3auto"] | true;
      PR3relayState = doc["PR3etatMan"] | false;

      PR4heureDebut = doc["PR4debut"] | "11:00";
      PR4heureFin = doc["PR4fin"] | "21:00";
      PR4modeAuto = doc["PR4auto"] | true;
      PR4relayState = doc["PR4etatMan"] | false;

      Serial.println("Paramètres chargés depuis LittleFS");
    }
    file.close();
  }
}

void setup() {
  Serial.begin(115200);
  pinMode(PR1RELAY_PIN, OUTPUT);
  pinMode(PR2RELAY_PIN, OUTPUT);
  pinMode(PR3RELAY_PIN, OUTPUT);
  pinMode(PR4RELAY_PIN, OUTPUT);

  // Initialisation LittleFS
  if (!LittleFS.begin(true)) {
    Serial.println("Erreur critique LittleFS");
    return;
  }

  loadSettings();  // Charger les valeurs au démarrage
  digitalWrite(PR1RELAY_PIN, PR1relayState ? HIGH : LOW);
  digitalWrite(PR2RELAY_PIN, PR2relayState ? HIGH : LOW);
  digitalWrite(PR3RELAY_PIN, PR3relayState ? HIGH : LOW);
  digitalWrite(PR4RELAY_PIN, PR4relayState ? HIGH : LOW);

  WiFi.begin(ssid, password);
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }

  Serial.println("");
  Serial.println("WiFi connected.");
  Serial.println("IP address: ");
  // Imprimer l'adresse IP locale ESP
  Serial.println(WiFi.localIP());

  configTzTime(TZ_INFO, "pool.ntp.org", "time.google.com");

  // --- ROUTES WEB ---
  server.on("/", HTTP_GET, [](AsyncWebServerRequest *request) {
    request->send(LittleFS, "/index.html", "text/html");
  });

  server.on("/get-data", HTTP_GET, [](AsyncWebServerRequest *request) {
    StaticJsonDocument<400> doc;
    doc["PR1debut"] = PR1heureDebut;
    doc["PR1fin"] = PR1heureFin;
    doc["PR1auto"] = PR1modeAuto;
    doc["PR1boolEtat"] = PR1relayState;

    doc["PR2debut"] = PR2heureDebut;
    doc["PR2fin"] = PR2heureFin;
    doc["PR2auto"] = PR2modeAuto;
    doc["PR2boolEtat"] = PR2relayState;

    doc["PR3debut"] = PR3heureDebut;
    doc["PR3fin"] = PR3heureFin;
    doc["PR3auto"] = PR3modeAuto;
    doc["PR3boolEtat"] = PR3relayState;

    doc["PR4debut"] = PR4heureDebut;
    doc["PR4fin"] = PR4heureFin;
    doc["PR4auto"] = PR4modeAuto;
    doc["PR4boolEtat"] = PR4relayState;

    struct tm timeinfo;
    char buff[10];
    if (getLocalTime(&timeinfo)) strftime(buff, sizeof(buff), "%H:%M:%S", &timeinfo);
    else sprintf(buff, "--:--:--");
    doc["actuelle"] = String(buff);

    String json;
    serializeJson(doc, json);
    request->send(200, "application/json", json);
  });

  server.on("/PR1toggle-mode", HTTP_GET, [](AsyncWebServerRequest *request) {
    PR1modeAuto = !PR1modeAuto;
    saveSettings();  // Sauvegarde le changement de mode
    request->send(200, "text/plain", "OK");
  });

  server.on("/PR2toggle-mode", HTTP_GET, [](AsyncWebServerRequest *request) {
    PR2modeAuto = !PR2modeAuto;
    saveSettings();  // Sauvegarde le changement de mode
    request->send(200, "text/plain", "OK");
  });

  server.on("/PR3toggle-mode", HTTP_GET, [](AsyncWebServerRequest *request) {
    PR3modeAuto = !PR3modeAuto;
    saveSettings();  // Sauvegarde le changement de mode
    request->send(200, "text/plain", "OK");
  });

  server.on("/PR4toggle-mode", HTTP_GET, [](AsyncWebServerRequest *request) {
    PR4modeAuto = !PR4modeAuto;
    saveSettings();  // Sauvegarde le changement de mode
    request->send(200, "text/plain", "OK");
  });

  server.on("/PR1force-state", HTTP_GET, [](AsyncWebServerRequest *request) {
    PR1modeAuto = false;
    PR1relayState = !PR1relayState;
    saveSettings();  // Sauvegarde l'état forcé
    request->send(200, "text/plain", "PR1OK");
  });

  server.on("/PR2force-state", HTTP_GET, [](AsyncWebServerRequest *request) {
    PR2modeAuto = false;
    PR2relayState = !PR2relayState;
    Serial.println("test");
    saveSettings();  // Sauvegarde l'état forcé
    request->send(200, "text/plain", "PR2OK");
  });

  server.on("/PR3force-state", HTTP_GET, [](AsyncWebServerRequest *request) {
    PR3modeAuto = false;
    PR3relayState = !PR3relayState;
    saveSettings();  // Sauvegarde l'état forcé
    request->send(200, "text/plain", "PR3OK");
  });

  server.on("/PR4force-state", HTTP_GET, [](AsyncWebServerRequest *request) {
    PR4modeAuto = false;
    PR4relayState = !PR4relayState;
    saveSettings();  // Sauvegarde l'état forcé
    request->send(200, "text/plain", "PR4OK");
  });

  server.on("/PR1save", HTTP_POST, [](AsyncWebServerRequest *request) {
    if (request->hasParam("PR1debut", true) && request->hasParam("PR1fin", true)) {
      PR1heureDebut = request->getParam("PR1debut", true)->value();
      PR1heureFin = request->getParam("PR1fin", true)->value();
      saveSettings();  // Sauvegarde les nouveaux horaires
      request->send(200, "text/plain", "OK");
    }
  });

  server.on("/PR2save", HTTP_POST, [](AsyncWebServerRequest *request) {
    if (request->hasParam("PR2debut", true) && request->hasParam("PR2fin", true)) {
      PR2heureDebut = request->getParam("PR2debut", true)->value();
      PR2heureFin = request->getParam("PR2fin", true)->value();
      saveSettings();  // Sauvegarde les nouveaux horaires
      request->send(200, "text/plain", "OK");
    }
  });

  server.on("/PR3save", HTTP_POST, [](AsyncWebServerRequest *request) {
    if (request->hasParam("PR3debut", true) && request->hasParam("PR3fin", true)) {
      PR3heureDebut = request->getParam("PR3debut", true)->value();
      PR3heureFin = request->getParam("PR3fin", true)->value();
      saveSettings();  // Sauvegarde les nouveaux horaires
      request->send(200, "text/plain", "OK");
    }
  });

  server.on("/PR4save", HTTP_POST, [](AsyncWebServerRequest *request) {
    if (request->hasParam("PR4debut", true) && request->hasParam("PR4fin", true)) {
      PR4heureDebut = request->getParam("PR4debut", true)->value();
      PR4heureFin = request->getParam("PR4fin", true)->value();
      saveSettings();  // Sauvegarde les nouveaux horaires
      request->send(200, "text/plain", "OK");
    }
  });

  server.begin();
}

void loop() {

  static unsigned long lastCheck = 0;
  if (millis() - lastCheck >= 1000) {
    lastCheck = millis();

    struct tm timeinfo;
    if (getLocalTime(&timeinfo)) {
      char nowStr[6];
      strftime(nowStr, sizeof(nowStr), "%H:%M", &timeinfo);
      now = String(nowStr);
    }

    if (PR1modeAuto) {
      bool PR1newState;
      if (PR1heureDebut < PR1heureFin) PR1newState = (now >= PR1heureDebut && now < PR1heureFin);
      else PR1newState = (now >= PR1heureDebut || now < PR1heureFin);

      if (PR1newState != PR1relayState) {
        PR1relayState = PR1newState;
        digitalWrite(PR1RELAY_PIN, PR1relayState ? HIGH : LOW);
      }
    } else {
      digitalWrite(PR1RELAY_PIN, PR1relayState ? HIGH : LOW);
    }

    if (PR2modeAuto) {
      bool PR2newState;
      if (PR2heureDebut < PR2heureFin) PR2newState = (now >= PR2heureDebut && now < PR2heureFin);
      else PR2newState = (now >= PR2heureDebut || now < PR2heureFin);

      if (PR2newState != PR2relayState) {
        PR2relayState = PR2newState;
        digitalWrite(PR2RELAY_PIN, PR2relayState ? HIGH : LOW);
      }
    } else {
      digitalWrite(PR2RELAY_PIN, PR2relayState ? HIGH : LOW);
    }

    if (PR3modeAuto) {
      bool PR3newState;
      if (PR3heureDebut < PR3heureFin) PR3newState = (now >= PR3heureDebut && now < PR3heureFin);
      else PR3newState = (now >= PR3heureDebut || now < PR3heureFin);

      if (PR3newState != PR3relayState) {
        PR3relayState = PR3newState;
        digitalWrite(PR3RELAY_PIN, PR3relayState ? HIGH : LOW);
      }
    } else {
      digitalWrite(PR3RELAY_PIN, PR3relayState ? HIGH : LOW);
    }

    if (PR4modeAuto) {
      bool PR4newState;
      if (PR4heureDebut < PR4heureFin) PR4newState = (now >= PR4heureDebut && now < PR4heureFin);
      else PR4newState = (now >= PR4heureDebut || now < PR4heureFin);

      if (PR4newState != PR4relayState) {
        PR4relayState = PR4newState;
        digitalWrite(PR4RELAY_PIN, PR4relayState ? HIGH : LOW);
      }
    } else {
      digitalWrite(PR4RELAY_PIN, PR4relayState ? HIGH : LOW);
    }
  }
}