import processing.serial.*;
import java.util.ArrayList;

// Classe pour stocker l'angle, la distance et le temps de vie de chaque point
class ScanPoint {
  float angle;
  float distance;
  int lifetime; // Temps de vie restant (en frames)

  ScanPoint(float a, float d, int lt) {
    angle = a;
    distance = d;
    lifetime = lt;
  }
}

// L'objet Serial pour la communication
Serial myPort;
String data = "";

// Variables pour stocker les données lues
float currentAngle = 0;
float currentDistance = 0;
float currentSpeed = 0; 

// Constantes pour les limites de distance (en mm)
final float MIN_DIST = 10; // 10 mm (Distance minimale mesurable)
final float MAX_DIST = 2000; // 1000 mm (Distance maximale mesurable)

// Rayon maximal pour l'affichage (en pixels)
final float MAX_DRAW_RADIUS = 300; 

// Centre du radar
float centerX, centerY;

// Liste pour stocker tous les points scannés avec leur temps de vie
ArrayList<ScanPoint> scanHistory = new ArrayList<ScanPoint>();
// Les points vivent pendant 100 frames (pour un effacement plus long)
final int POINT_LIFETIME = 100; 
// Opacité du fondu d'arrière-plan (plus petit = plus de rémanence)
final int FADE_OPACITY = 10; 


void setup() {
  size(1050, 700); 
  smooth(); 
  frameRate(60); 
  
  // Le centre est décalé vers la gauche
  // Ajustement de centerX pour laisser de la place aux étiquettes d'angle
  centerX = (height - 50) / 2 + 30; 
  centerY = height / 2;

  println(Serial.list());
  
  // --- CONFIGURATION DU PORT SÉRIE ---
  try {
    myPort = new Serial(this, "COM4", 115200); 
    myPort.bufferUntil('\n'); 
  } catch (Exception e) {
    println("Erreur d'ouverture du port série: " + e.getMessage());
    println("Vérifiez le nom du port et s'il est déjà utilisé.");
  }
  background(0); // Fond noir initial
}

void draw() {
  // 1. Mise à jour de l'arrière-plan (effet de rémanence)
  fill(0, FADE_OPACITY); 
  rect(0, 0, width, height);

  pushMatrix();
  translate(centerX, centerY);

  // 2. Dessin de l'échelle du radar (lignes de grille)
  noFill();
  stroke(50, 200, 50, 150); // Vert foncé pour la grille
  strokeWeight(1);
  
  // Cercles de distance
  ellipse(0, 0, MAX_DRAW_RADIUS * 2, MAX_DRAW_RADIUS * 2); 
  float halfRadius = map(500, MIN_DIST, MAX_DIST, 0, MAX_DRAW_RADIUS);
  ellipse(0, 0, halfRadius * 2, halfRadius * 2);
  float minRadius = map(MIN_DIST, MIN_DIST, MAX_DIST, 0, MAX_DRAW_RADIUS);
  ellipse(0, 0, minRadius * 2, minRadius * 2); 
  
  // Étiquettes de distance
  fill(50, 200, 50);
  textSize(12);
  textAlign(CENTER);
  text("1000 mm", MAX_DRAW_RADIUS - 30, -5);
  text("500 mm", halfRadius - 20, -5);
  text("10 mm", minRadius - 15, -5);
  
  // Lignes d'angle (tous les 45 degrés)
  stroke(50, 200, 50, 100);
  for (int a = 0; a < 360; a += 45) {
    float x_line = MAX_DRAW_RADIUS * cos(radians(a));
    float y_line = MAX_DRAW_RADIUS * sin(radians(a));
    line(0, 0, x_line, y_line);
  }
  
  // MISE À JOUR : AFFICHAGE DE L'ÉCHELLE DES ANGLES (de 0° à 360° complet)
  float angleTextRadius = MAX_DRAW_RADIUS + 25; // Rayon légèrement plus grand pour le texte
  textSize(14);
  fill(0, 200, 0); // Vert pour les étiquettes d'angle

  // Boucle de 0 à 359 par pas de 30 degrés. 
  // Remarque : 360° est la même position que 0°, on affiche donc 0° une seule fois.
  for (int a = 0; a < 360; a += 30) { 
    // Utiliser 0° pour l'affichage à la place de 360° pour la lisibilité
    int displayAngle = a == 360 ? 0 : a;
    
    // Calculer la position (x, y) pour le texte sur le cercle extérieur
    float x_text = angleTextRadius * cos(radians(a));
    float y_text = angleTextRadius * sin(radians(a));
    
    // Ajuster l'alignement pour que le texte soit centré sur le point
    if (a == 0) textAlign(LEFT); // 0°/360° est à droite, on décale le texte à gauche du point
    else if (a == 180) textAlign(RIGHT); // 180° est à gauche, on décale le texte à droite du point
    else textAlign(CENTER); // Ailleurs : centrer
    
    // Afficher l'angle
    text(displayAngle + "°", x_text, y_text);
  }

  // 3. --- DESSIN ET MISE À JOUR DE L'HISTORIQUE DES POINTS ---
  
  ArrayList<ScanPoint> updatedHistory = new ArrayList<ScanPoint>();

  for (int i = 0; i < scanHistory.size(); i++) {
    ScanPoint p = scanHistory.get(i);
    
    // Calculer la position et la transparence
    float drawRadius = map(p.distance, MIN_DIST, MAX_DIST, 0, MAX_DRAW_RADIUS);
    float angleInRadians = radians(p.angle);
    float x = drawRadius * cos(angleInRadians);
    float y = drawRadius * sin(angleInRadians);
    
    // Transparence basée sur le temps de vie (alpha max 255)
    float alpha = map(p.lifetime, 0, POINT_LIFETIME, 0, 255);
    
    // Dessiner le point
    stroke(255, 0, 0, alpha); // Rouge, avec transparence
    strokeWeight(3);
    point(x, y);

    // Mettre à jour le temps de vie
    p.lifetime--;
    
    // Garder le point s'il a encore du temps de vie
    if (p.lifetime > 0) {
      updatedHistory.add(p);
    }
  }
  
  scanHistory = updatedHistory; // Mettre à jour l'historique

  // 4. Dessin de la ligne de balayage actuelle (vert vif)
  float currentDrawRadius = map(currentDistance, MIN_DIST, MAX_DIST, 0, MAX_DRAW_RADIUS);
  float currentAngleInRadians = radians(currentAngle); 
  float currentX = currentDrawRadius * cos(currentAngleInRadians);
  float currentY = currentDrawRadius * sin(currentAngleInRadians);
  
  stroke(0, 255, 0, 255); // Vert vif (totalement opaque)
  strokeWeight(2);
  line(0, 0, currentX, currentY);

  popMatrix();
  
  // 5. --- AFFICHAGE DE LA LÉGENDE ET DES VALEURS ---
  
  float legendX = 2 * centerX + 10; 
  float legendY = 50;
  
  fill(255);
  textSize(24);
  textAlign(LEFT);
  text("LÉGENDE ET DONNÉES", legendX, legendY);
  
  // Section des données actuelles
  textSize(16);
  text("Données Actuelles:", legendX, legendY + 40);
  fill(0, 255, 0); // Vert pour les données en direct
  text("Angle (θ): " + nf(currentAngle, 0, 2) + "°", legendX, legendY + 65);
  text("Distance (D): " + nf(currentDistance, 0, 2) + " mm", legendX, legendY + 90);
  text("Vitesse (V): " + nf(currentSpeed, 0, 2) + " m/s", legendX, legendY + 115);
  
  // Séparateur 
  stroke(255);
  line(legendX, legendY + 135, legendX + 250, legendY + 135);
  
  // Section de la Légende Graphique 
  fill(255);
  text("Légende Graphique:", legendX, legendY + 165);
  
  // Ligne de balayage actuelle
  stroke(0, 255, 0, 255);
  strokeWeight(2);
  line(legendX, legendY + 190, legendX + 20, legendY + 190);
  fill(255);
  text("Balayage/Mesure Actuelle", legendX + 30, legendY + 195);
  
  // Point historique
  stroke(255, 0, 0, 255);
  strokeWeight(3);
  point(legendX + 10, legendY + 225);
  fill(255);
  text("Point Historique (s'estompe)", legendX + 30, legendY + 230);

  // Échelle de distance
  stroke(50, 200, 50, 150);
  strokeWeight(1);
  noFill();
  ellipse(legendX + 10, legendY + 260, 20, 20);
  fill(255);
  text("Grille de Distance (10-1000 mm)", legendX + 30, legendY + 265);
}

void serialEvent(Serial p) {
  // Lecture et parsing des données 
  try {
    data = p.readStringUntil('\n'); 
  } catch (Exception e) {
    return;
  }
  
  if (data != null) {
    data = trim(data); 

    if (data.contains(",")) {
      String[] values = split(data, ',');
      
      if (values.length == 3) {
        try {
          currentAngle = float(values[0]);
          currentDistance = float(values[1]);
          currentSpeed = float(values[2]); 
          
          currentAngle = currentAngle % 360;
          if (currentAngle < 0) {
            currentAngle += 360;
          }
          currentDistance = constrain(currentDistance, MIN_DIST, MAX_DIST);
          
          // Ajouter le point à l'historique
          ScanPoint newPoint = new ScanPoint(currentAngle, currentDistance, POINT_LIFETIME);
          scanHistory.add(newPoint);
          
        } catch (NumberFormatException e) {
          println("Erreur de format de nombre: " + data);
        }
      } 
    }
  }
}
